Attribute VB_Name = "Simulation"
' Set of geometric dimensions
Public yokeWidth As Double
Public yokeHeight As Double
Public maghetWidth As Double
Public magnetHeight As Double
Public keeperWidth As Double
Public keeperHeight As Double
Public airGap As Double
' Magnet coercive force
Public Hc As Double

' Stored QF pictures to show in the MainDialog
Public geomPicture As StdPicture
Public resultPicture As StdPicture
Public sketchPicture As StdPicture

Private Const PI As Double = 3.1415926

' The most common QuickField objects: Application and Problem
Private QF As QuickField.Application
Private prb As QuickField.Problem

Public Sub DoCalculation()
    Dim force As Double
    
    Set QF = CreateObject("QuickField.Application")
    QF.DefaultFilePath = VB.App.Path & "\Magn1"
    'QF.MainWindow.WindowState = qfMaximized
    CreateProblem
    BuildGeometry
    SetData
    Solve
    
    ViewResults
    force = CalculateForce()
    MainDialog.DisplayForce force
    
    QF.Quit
    Set QF = Nothing
End Sub

Sub CreateProblem()
    Set prb = QF.Problems.Add
    With prb
        .ProblemType = qfMagnetostatics
        .Class = qfPlaneParallel
        .LengthUnits = qfCentimeters
        .Coordinates = qfCartesian
        .ReferencedFile(qfModelFile) = "Magn1.mod"
        .ReferencedFile(qfDataFile) = "Magn1.dms"
        .SaveAs "Magn1.pbm"  'Save the new problem
    End With
End Sub

Sub BuildGeometry()
    Dim mdl As QuickField.Model
    Dim shp As QuickField.ShapeRange
    Dim yBase As Double
    Dim xBase As Double
    
    Set mdl = QF.Models.Add
    mdl.SaveAs prb.ReferencedFile(qfModelFile)
    With mdl.Shapes
            ' the yoke
        Set shp = .AddEdge(QF.PointXY(-yokeWidth / 2, 0), QF.PointXY(yokeWidth / 2, 0))
        .AddEdge QF.PointXY(yokeWidth / 2, 0), QF.PointXY(yokeWidth / 2, yokeHeight)
        .AddEdge QF.PointXY(yokeWidth / 2, yokeHeight), QF.PointXY(-yokeWidth / 2, yokeHeight)
        .AddEdge QF.PointXY(-yokeWidth / 2, yokeHeight), QF.PointXY(-yokeWidth / 2, 0)
        shp.Left.Label = "Steel"
            ' Right Magnet
        Set shp = .AddEdge(QF.PointXY(yokeWidth / 2, yokeHeight), _
                           QF.PointXY(yokeWidth / 2, yokeHeight + magnetHeight))
        .AddEdge QF.PointXY(yokeWidth / 2, yokeHeight + magnetHeight), _
                 QF.PointXY(yokeWidth / 2 - maghetWidth, yokeHeight + magnetHeight)
        .AddEdge QF.PointXY(yokeWidth / 2 - maghetWidth, yokeHeight + magnetHeight), _
                 QF.PointXY(yokeWidth / 2 - maghetWidth, yokeHeight)
        shp.Left.Label = "ALNICO down"
            ' Left Magnet
        Set shp = .AddEdge(QF.PointXY(-yokeWidth / 2, yokeHeight), _
                           QF.PointXY(-yokeWidth / 2, yokeHeight + magnetHeight))
        .AddEdge QF.PointXY(-yokeWidth / 2, yokeHeight + magnetHeight), _
                 QF.PointXY(-yokeWidth / 2 + maghetWidth, yokeHeight + magnetHeight)
        .AddEdge QF.PointXY(-yokeWidth / 2 + maghetWidth, yokeHeight + magnetHeight), _
                 QF.PointXY(-yokeWidth / 2 + maghetWidth, yokeHeight)
        shp.Right.Label = "ALNICO up"
            ' Steel Keeper
        yBase = yokeHeight + magnetHeight + airGap
        Set shp = .AddEdge(QF.PointXY(-keeperWidth / 2, yBase), _
                            QF.PointXY(keeperWidth / 2, yBase))
        .AddEdge QF.PointXY(keeperWidth / 2, yBase), _
                 QF.PointXY(keeperWidth / 2, yBase + keeperHeight)
        .AddEdge QF.PointXY(keeperWidth / 2, yBase + keeperHeight), _
                 QF.PointXY(-keeperWidth / 2, yBase + keeperHeight)
        .AddEdge QF.PointXY(-keeperWidth / 2, yBase + keeperHeight), _
                 QF.PointXY(-keeperWidth / 2, yBase)
        shp.Left.Label = "Steel Keeper"
            ' Surrounding air
        yBase = yokeHeight + magnetHeight + airGap + keeperHeight
        xBase = (yokeWidth + keeperWidth) * 1.5
        Set shp = .AddEdge(QF.PointXY(-xBase, -yBase), QF.PointXY(xBase, -yBase))
        .AddEdge QF.PointXY(xBase, -yBase), QF.PointXY(xBase, 2 * yBase)
        .AddEdge QF.PointXY(xBase, 2 * yBase), QF.PointXY(-xBase, 2 * yBase)
        .AddEdge QF.PointXY(-xBase, 2 * yBase), QF.PointXY(-xBase, -yBase)
        shp.Left.Label = "Air"
        .Boundary(qfOuterOnly).Label = "Zero"
        
            ' Set Spacing
        Dim spCore As Double        ' Average spacing inside magnets and aramature
        Dim spSurround As Double    ' Average spacing in surroundng air
        
        spCore = fMax(yokeHeight + magnetHeight + airGap + keeperHeight, _
                       (keeperWidth + yokeWidth) / 2)
        spCore = spCore / 8
        spSurround = (xBase + yBase) / 6
        
        '.LabeledAs(Block:="Steel Keeper").Spacing = sp
        '.LabeledAs(Block:="Steel").Spacing = sp
        .LabeledAs(Block:="ALNICO up").Spacing = spCore
        .LabeledAs(Block:="ALNICO down").Spacing = spCore
        .LabeledAs(Edge:="Zero").Spacing = spSurround
            ' Generate the mesh
        .BuildMesh
    End With
    
        ' Store the picture to show in the MainDialog
    Dim win As QuickField.ModelWindow
    Set win = mdl.Windows(1)
    win.WindowState = qfNormal
    win.Height = win.Width + 10
    win.Zoom
    win.GetPicture
    'Set geomPicture = Clipboard.GetData(vbCFMetafile)
    Set geomPicture = Clipboard.GetData
        
        ' Save the complete model
    mdl.Save
End Sub

Sub SetData()
    Dim lab As QuickField.Label
    Dim elem As Variant
    
    '   First set properties for block labels
    Dim cntBlock As QuickField.LabelBlockMS
    For Each elem In prb.Labels(qfBlock)
        Set lab = elem
        Set cntBlock = lab.Content
        Select Case lab.Name
            Case "Air"
                cntBlock.Kxx = 1
                cntBlock.Kyy = 1
            
            Case "Steel", "Steel Keeper"
                Dim spl As QuickField.Spline
                Set spl = cntBlock.CreateBHCurve
                spl.Add QF.PointXY(0.73, 400)
                spl.Add QF.PointXY(0.92, 600)
                spl.Add QF.PointXY(1.05, 800)
                spl.Add QF.PointXY(1.15, 1000)
                spl.Add QF.PointXY(1.28, 1400)
                spl.Add QF.PointXY(1.42, 2000)
                spl.Add QF.PointXY(1.52, 3000)
                spl.Add QF.PointXY(1.58, 4000)
                spl.Add QF.PointXY(1.6, 6000)
                cntBlock.Spline = spl
                
            Case "ALNICO up", "ALNICO down"
                If StrComp(lab.Name, "ALNICO up") = 0 Then
                    cntBlock.Coercive = QF.PointRA(Hc, PI / 2)
                Else
                    cntBlock.Coercive = QF.PointRA(Hc, -PI / 2)
                End If
                Set spl = cntBlock.CreateBHCurve
                spl.Add QF.PointXY(0.24, 27818 - Hc)
                spl.Add QF.PointXY(0.4, 47748 - Hc)
                spl.Add QF.PointXY(0.5, 67641 - Hc)
                spl.Add QF.PointXY(0.6, 93504 - Hc)
                spl.Add QF.PointXY(0.71, 127324 - Hc)
                spl.Add QF.PointXY(0.77, 147218 - Hc)
                cntBlock.Spline = spl
        End Select
        lab.Content = cntBlock
    Next
    
    '   The only edge label
    Dim cntEdge As QuickField.LabelEdgeMS
    Set cntEdge = prb.Labels(qfEdge).Item("Zero").Content
    cntEdge.Dirichlet = 0
    prb.Labels(qfEdge).Item("Zero").Content = cntEdge
    '   Saving data document
    prb.DataDoc.Save
End Sub

Sub Solve()
    If prb.CanSolve Then prb.SolveProblem
    If prb.Solved Then prb.AnalyzeResults
End Sub
    
Sub ViewResults()
    Dim res As QuickField.Result
    Set res = prb.Result
    If res Is Nothing Then Exit Sub
    
    Dim win As QuickField.FieldWindow
    Set win = res.Windows(1)
    win.WindowState = qfNormal
    win.Height = win.Width + 10
    win.Zoom
    win.GetPicture
    'Set resultPicture = Clipboard.GetData(vbCFMetafile)
    Set resultPicture = Clipboard.GetData
End Sub

Function CalculateForce() As Double
    Dim res As QuickField.Result
    Set res = prb.Result
    If res Is Nothing Then Exit Function
    
    Dim win As QuickField.FieldWindow
    Set win = res.Windows(1)
    With win.Contour
        .AddLineTo QF.PointXY(-keeperWidth / 2 - maghetWidth, yokeHeight + magnetHeight + airGap / 2)
        .AddLineTo QF.PointXY(keeperWidth / 2 + maghetWidth, yokeHeight + magnetHeight + airGap / 2)
        .AddLineTo QF.PointXY(keeperWidth / 2 + maghetWidth, yokeHeight + magnetHeight + airGap + keeperHeight * 1.5)
        .AddLineTo QF.PointXY(-keeperWidth / 2 - maghetWidth, yokeHeight + magnetHeight + airGap + keeperHeight * 1.5)
        .AddLineTo QF.PointXY(-keeperWidth / 2 - maghetWidth, yokeHeight + magnetHeight + airGap / 2)
    End With
    CalculateForce = res.GetIntegral(qfInt_MaxwellForce).Abs
    win.Contour.Delete True
End Function

Private Function fMin(f1 As Double, f2 As Double) As Double
    If f1 >= f2 Then
        fMin = f2
    Else
        fMin = f1
    End If
End Function

Private Function fMax(f1 As Double, f2 As Double) As Double
    If f1 >= f2 Then
        fMax = f1
    Else
        fMax = f2
    End If
End Function

